//
// Copyright (C) 2020, NinjaTrader LLC <www.ninjatrader.com>.
// NinjaTrader reserves the right to modify or overwrite this NinjaScript component with each release.
//
#region Using declarations
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using System.Windows.Media;
using System.Xml.Serialization;
using NinjaTrader.Cbi;
using NinjaTrader.Gui;
using NinjaTrader.Gui.Chart;
using NinjaTrader.Gui.SuperDom;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.Core.FloatingPoint;
using NinjaTrader.NinjaScript.DrawingTools;
#endregion

// This namespace holds indicators in this folder and is required. Do not change it.
namespace NinjaTrader.NinjaScript.Indicators.ZombiePack9
{
	/// <summary>
	/// The MACD (Moving Average Convergence/Divergence) is a trend following momentum indicator
	/// that shows the relationship between two moving averages of prices.
	/// </summary>
	public class Zombie9MACD : Indicator
	{
		private const string SystemVersion = "v1.072";
		private const string SystemName = "Zombie9MACD";
		private const string FullSystemName = SystemName + " - " + SystemVersion;

		private	Series<double>		macdEMAFast;
		private Series<double> macdEMASlow;

		private double				constant1;
		private double				constant2;
		private double				constant3;
		private double				constant4;
		private double				constant5;
		private double				constant6;

		const int LineChangePlotIndex = 0;
		const int LineBullishPlotIndex = 1;
		const int LineBearishPlotIndex = 2;

		const int AvgChangePlotIndex = 3;
		const int AvgBullishPlotIndex = 4;
		const int AvgBearishPlotIndex = 5;

		private Brush lineChangeColor;
		private Brush lineBullishColor;
		private Brush lineBearishColor;

		private Brush avgChangeColor;
		private Brush avgBullishColor;
		private Brush avgBearishColor;

		public override string DisplayName
		{
			get
			{
				if (State == State.SetDefaults)
					return FullSystemName;
				else if (ShowIndicatorName)
					return FullSystemName;
				else
					return "";
			}
		}

		protected override void OnStateChange()
		{
			if (State == State.SetDefaults)
			{
				Name = SystemName;
				Description = FullSystemName;
				Calculate = Calculate.OnPriceChange;
				PaintPriceMarkers = false; 
				IsSuspendedWhileInactive = true;
				ScaleJustification = ScaleJustification.Overlay;
				BarsToLoad = 2;
				MACDFastPeriod						= 12;
				MACDSlowPeriod						= 26;
				MACDSmoothPeriod = 9;


				//levels
				//AddLine(Brushes.DarkGray, 0, NinjaTrader.Custom.Resource.NinjaScriptIndicatorZeroLine);


				//lines
				AddPlot(new Stroke(Brushes.DimGray, 2), PlotStyle.Line, "LineChange"); 
				AddPlot(new Stroke(Brushes.DimGray, 2), PlotStyle.Line, "LineBullish"); 
				AddPlot(new Stroke(Brushes.DimGray, 2), PlotStyle.Line, "LineBearish"); 

				AddPlot(new Stroke(Brushes.DimGray, 3), PlotStyle.Line, "AvgChange");
				AddPlot(new Stroke(Brushes.RoyalBlue, 3), PlotStyle.Line, "AvgBullish"); 
				AddPlot(new Stroke(Brushes.Magenta, 3), PlotStyle.Line, "AvgBearish");

			}
			else if (State == State.Configure)
			{
				constant1	= 2.0 / (1 + MACDFastPeriod);
				constant2	= (1 - (2.0 / (1 + MACDFastPeriod)));
				constant3	= 2.0 / (1 + MACDSlowPeriod);
				constant4	= (1 - (2.0 / (1 + MACDSlowPeriod)));
				constant5	= 2.0 / (1 + MACDSmoothPeriod);
				constant6	= (1 - (2.0 / (1 + MACDSmoothPeriod)));
			}
			else if (State == State.DataLoaded)
			{
				macdEMAFast = new Series<double>(this);
				macdEMASlow = new Series<double>(this);

				lineChangeColor = Plots[LineChangePlotIndex].Brush;
				lineBullishColor = Plots[LineBullishPlotIndex].Brush;
				lineBearishColor = Plots[LineBearishPlotIndex].Brush;

				avgChangeColor = Plots[AvgChangePlotIndex].Brush;
				avgBullishColor = Plots[AvgBullishPlotIndex].Brush;
				avgBearishColor = Plots[AvgBearishPlotIndex].Brush;
			}
		}

		protected override void OnBarUpdate()
		{
			if (CurrentBar < MACDSlowPeriod)
				return;

			double input0	= Input[0];
			double input1 = Input[1];


			if (CurrentBar == 0)
			{
				macdEMAFast[0]		= input0;
				macdEMASlow[0]		= input0;
				LineChange[0]		= 0;
				LineBullish[0] = 0;
				LineBearish[0] = 0;
				AvgChange[0] = 0;
				AvgBullish[0] = 0;
				AvgBearish[0] = 0;

			}
			else
			{
				double fastEma0	= constant1 * input0 + constant2 * macdEMAFast[1];
				double slowEma0	= constant3 * input0 + constant4 * macdEMASlow[1];
				double fastEma1 = constant1 * input1 + constant2 * macdEMAFast[2];
				double slowEma1 = constant3 * input1 + constant4 * macdEMASlow[2];
				double macd		= fastEma0 - slowEma0;
				double previousMACD = fastEma1 - slowEma1;
				double macdAvg	= constant5 * macd + constant6 * AvgChange[1];
				double previousMACDAvg = constant5 * previousMACD + constant6 * AvgChange[2];
		


				macdEMAFast[0]		= fastEma0;
				macdEMASlow[0]		= slowEma0;
				LineChange[0]		= macd;
				AvgChange[0]			= macdAvg;

				bool bullishMACDSlope = (macd >= macdAvg); //(macd >= previousMACD);

				if (bullishMACDSlope)
                {
					PlotBrushes[LineChangePlotIndex][0] = lineBullishColor;
					//LineBullish[0] = macd;
				}
				else
                {
					PlotBrushes[LineChangePlotIndex][0] = lineBearishColor;
					//LineBearish[0] = macd;
				}
				

				bool bullishAvgSlope =  (macdAvg >= previousMACDAvg);

				if (bullishAvgSlope)
				{
					PlotBrushes[AvgChangePlotIndex][0] = avgBullishColor;
					//AvgBullish[0] = macdAvg;
				}
				else
				{
					PlotBrushes[AvgChangePlotIndex][0] = avgBearishColor;
					//AvgBearish[0] = macdAvg;
				}
				

			}
		}

		#region Properties

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> LineChange
		{
			get { return Values[0]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> LineBullish
		{
			get { return Values[1]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> LineBearish
		{
			get { return Values[2]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> AvgChange
		{
			get { return Values[3]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> AvgBullish
		{
			get { return Values[4]; }
		}

		[Browsable(false)]
		[XmlIgnore]
		public Series<double> AvgBearish
		{
			get { return Values[5]; }
		}


		

		[NinjaScriptProperty]
		[Display(Name = "IndicatorName", GroupName = "0) Indicator Information", Order = 0)]
		public string IndicatorName
		{
			get { return FullSystemName; }
			set { }
		}

		[NinjaScriptProperty]
		[Display(Name = "ShowIndicatorName", GroupName = "0) Indicator Information", Order = 1)]
		public bool ShowIndicatorName
		{ get; set; }

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), GroupName = "Parameters", Order = 0)]
		public int MACDFastPeriod
		{ get; set; }

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), GroupName = "Parameters", Order = 1)]
		public int MACDSlowPeriod
		{ get; set; }

		[Range(1, int.MaxValue), NinjaScriptProperty]
		[Display(ResourceType = typeof(Custom.Resource), GroupName = "Parameters", Order = 2)]
		public int MACDSmoothPeriod
		{ get; set; }

		#endregion
	}
}

#region NinjaScript generated code. Neither change nor remove.

namespace NinjaTrader.NinjaScript.Indicators
{
	public partial class Indicator : NinjaTrader.Gui.NinjaScript.IndicatorRenderBase
	{
		private ZombiePack9.Zombie9MACD[] cacheZombie9MACD;
		public ZombiePack9.Zombie9MACD Zombie9MACD(string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			return Zombie9MACD(Input, indicatorName, showIndicatorName, mACDFastPeriod, mACDSlowPeriod, mACDSmoothPeriod);
		}

		public ZombiePack9.Zombie9MACD Zombie9MACD(ISeries<double> input, string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			if (cacheZombie9MACD != null)
				for (int idx = 0; idx < cacheZombie9MACD.Length; idx++)
					if (cacheZombie9MACD[idx] != null && cacheZombie9MACD[idx].IndicatorName == indicatorName && cacheZombie9MACD[idx].ShowIndicatorName == showIndicatorName && cacheZombie9MACD[idx].MACDFastPeriod == mACDFastPeriod && cacheZombie9MACD[idx].MACDSlowPeriod == mACDSlowPeriod && cacheZombie9MACD[idx].MACDSmoothPeriod == mACDSmoothPeriod && cacheZombie9MACD[idx].EqualsInput(input))
						return cacheZombie9MACD[idx];
			return CacheIndicator<ZombiePack9.Zombie9MACD>(new ZombiePack9.Zombie9MACD(){ IndicatorName = indicatorName, ShowIndicatorName = showIndicatorName, MACDFastPeriod = mACDFastPeriod, MACDSlowPeriod = mACDSlowPeriod, MACDSmoothPeriod = mACDSmoothPeriod }, input, ref cacheZombie9MACD);
		}
	}
}

namespace NinjaTrader.NinjaScript.MarketAnalyzerColumns
{
	public partial class MarketAnalyzerColumn : MarketAnalyzerColumnBase
	{
		public Indicators.ZombiePack9.Zombie9MACD Zombie9MACD(string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			return indicator.Zombie9MACD(Input, indicatorName, showIndicatorName, mACDFastPeriod, mACDSlowPeriod, mACDSmoothPeriod);
		}

		public Indicators.ZombiePack9.Zombie9MACD Zombie9MACD(ISeries<double> input , string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			return indicator.Zombie9MACD(input, indicatorName, showIndicatorName, mACDFastPeriod, mACDSlowPeriod, mACDSmoothPeriod);
		}
	}
}

namespace NinjaTrader.NinjaScript.Strategies
{
	public partial class Strategy : NinjaTrader.Gui.NinjaScript.StrategyRenderBase
	{
		public Indicators.ZombiePack9.Zombie9MACD Zombie9MACD(string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			return indicator.Zombie9MACD(Input, indicatorName, showIndicatorName, mACDFastPeriod, mACDSlowPeriod, mACDSmoothPeriod);
		}

		public Indicators.ZombiePack9.Zombie9MACD Zombie9MACD(ISeries<double> input , string indicatorName, bool showIndicatorName, int mACDFastPeriod, int mACDSlowPeriod, int mACDSmoothPeriod)
		{
			return indicator.Zombie9MACD(input, indicatorName, showIndicatorName, mACDFastPeriod, mACDSlowPeriod, mACDSmoothPeriod);
		}
	}
}

#endregion
